
'use client';

import React from 'react';
import {
  Card,
  CardHeader,
  CardTitle,
  CardContent,
  CardDescription,
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import {
  MapPin,
  LoaderCircle,
  AlertTriangle,
  ArrowLeft,
  Home,
} from 'lucide-react';
import {
  useUser,
  useFirestore,
  useCollection,
  useMemoFirebase,
  useDoc
} from '@/firebase';
import { collection, doc } from 'firebase/firestore';
import type { Address, User } from '@/lib/data';
import { AddressCard } from '@/components/address-card';
import { Skeleton } from '@/components/ui/skeleton';
import { useParams, useRouter } from 'next/navigation';
import { Alert, AlertTitle, AlertDescription } from '@/components/ui/alert';

export default function UserAddressesPage() {
  const params = useParams();
  const router = useRouter();
  const userId = params.userId as string;
  const firestore = useFirestore();

  const userDocRef = useMemoFirebase(
    () => (userId ? doc(firestore, 'users', userId) : null),
    [firestore, userId]
  );
  
  const addressesCollectionRef = useMemoFirebase(
    () => (userId ? collection(firestore, 'users', userId, 'addresses') : null),
    [firestore, userId]
  );
  
  const { data: user, isLoading: isUserLoading } = useDoc<User>(userDocRef);
  const { data: addresses, isLoading: areAddressesLoading, error } = useCollection<Address>(addressesCollectionRef);

  const isLoading = isUserLoading || areAddressesLoading;

  return (
    <div className="p-4 md:p-8 space-y-6">
      <div className="flex items-center gap-4">
        <Button variant="ghost" size="icon" onClick={() => router.back()}>
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold">Address List</h1>
          {user ? (
            <p className="text-muted-foreground">
              Viewing all saved addresses for {user.name}.
            </p>
          ) : (
            <Skeleton className="h-5 w-48 mt-1" />
          )}
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <MapPin className="h-5 w-5 text-primary" />
            Saved Addresses
          </CardTitle>
          <CardDescription>
            A list of all shipping addresses associated with this user.
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading && (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <Skeleton className="h-40 w-full" />
              <Skeleton className="h-40 w-full" />
            </div>
          )}

          {!isLoading && error && (
            <Alert variant="destructive">
              <AlertTriangle className="h-4 w-4" />
              <AlertTitle>Error Loading Addresses</AlertTitle>
              <AlertDescription>
                Could not fetch address data. Please ensure you have the correct permissions.
              </AlertDescription>
            </Alert>
          )}

          {!isLoading && !error && addresses && addresses.length > 0 && (
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {addresses.map((address) => (
                <AddressCard key={address.id} address={address} />
              ))}
            </div>
          )}

          {!isLoading && !error && (!addresses || addresses.length === 0) && (
            <Alert>
              <Home className="h-4 w-4" />
              <AlertTitle>No Saved Addresses</AlertTitle>
              <AlertDescription>
                This user has not saved any addresses yet.
              </AlertDescription>
            </Alert>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
